#include "stdafx.h"
#include "DSP.h"
#include "RFDoc.h"

#include <cmath>

DSP::DSP()
{
	compBuf = NULL;
	window = NULL;
	fftSpec = NULL;
	cfftSpec = NULL;
	results = NULL;
	size = order = resultSize = 0;
}

DSP::~DSP()
{
	if(fftSpec) {
		ippsFFTFree_R_32f(fftSpec);
	}
	if(cfftSpec) {
		ippsFFTFree_C_32fc(cfftSpec);
	}
	if(window) {
		ippsFree(window);
	}
	if(compBuf) {
		ippsFree(compBuf);
	}
	if(results) {
		ippsFree(results);
	}
}

void DSP::Resize(int _order, bool sevenMeg) 
{
	if(fftSpec) {
		ippsFFTFree_R_32f(fftSpec);
		fftSpec = NULL;
	}
	if(cfftSpec) {
		ippsFFTFree_C_32fc(cfftSpec);
		cfftSpec = NULL;
	}
	if(window) {
		ippsFree(window);
		window = NULL;
	}
	if(compBuf) {
		ippsFree(compBuf);
		compBuf = NULL;
	}
	if(results) {
		ippsFree(results);
		results = NULL;
	}

	sevenMeg_ = sevenMeg;
	order = _order;
	size = pow2(order);
	if(sevenMeg_)
		resultSize = size;
	else 
		resultSize = size / 2;

	compBuf = ippsMalloc_32fc(size);
	window = ippsMalloc_32f(size);
	results = ippsMalloc_32f(resultSize);
	ippsZero_32f(results, resultSize);
	BuildWindow();

	if(sevenMeg_) {
		ippsFFTInitAlloc_C_32fc( 
			&cfftSpec, 
			order,
			IPP_FFT_DIV_FWD_BY_N, 
			ippAlgHintFast);	
	} else {
		ippsFFTInitAlloc_R_32f( 
			&fftSpec, 
			order,
			IPP_FFT_DIV_FWD_BY_N, 
			ippAlgHintFast);
	}
}

void DSP::Process(float *in)
{
	if(sevenMeg_) { 
		// 7 MHz processing
		ippsMul_32f32fc(window, (Ipp32fc*)in, compBuf, size);
		ippsFFTFwd_CToC_32fc_I(compBuf, cfftSpec, 0);
	
		// Convert to amp
		ippsPowerSpectr_32fc(compBuf, results + size/2, size/2);
		ippsPowerSpectr_32fc(compBuf + size/2, results, size/2);

	} else { 
		// 20 MHz processing
		ippsMul_32f_I(window, in, size);
		ippsFFTFwd_RToPerm_32f(in, (Ipp32f*)compBuf, fftSpec, 0);

		// Convert to amp
		ippsPowerSpectr_32fc(
			(Ipp32fc*)compBuf, // 4 instead of 2 because we doubled the fftSize
			results,
			resultSize);
	}
	
	// 10*log(10)
	ippsLn_32f_I(results, resultSize);
	ippsMulC_32f_I(4.3429448f, results, resultSize);

	//ippsAddC_32f_I(6.0, results, resultSize);
}

void DSP::BuildWindow()
{
	const float pi = 3.14159265359f;
	double avgval = 0.0;

	for(int k = 0; k < size; k++) {
		window[k] = 1 - 
			1.93*cos(2*pi*k/(size-1)) + 
			1.29*cos(4*pi*k/(size-1)) -
			0.388*cos(6*pi*k/(size-1)) +
			0.032*cos(8*pi*k/(size-1));
	}

	for(int k = 0; k < size; k++) {
		avgval += window[k];
	}

	double invavg = size / avgval;

	for(int k = 0; k < size; k++) {
		window[k] *= invavg;
	}
}

static void up_sample(float *src, float *dst, int count, int factor)
{
	int k = 0;
	for(int i = 0; i < count; i++) {
		for(int j = 0; j < factor; j++) {
			dst[k++] = src[i];
		}
	}
}

static void down_sample(float *src, float *dst, int count, int factor)
{
	int k = 0;
	for(int i = 0; i < count; i++) {
		dst[i] = src[k];
		k += factor;
	}
}

// Creates array of proper length
void DSP::Correct(float *corr) 
{
	int dstLen = 1024;
	if(resultSize == 2048) {
		ippsAdd_32f(results, corr, results, 2048);
	} else { // result != 2048
		float *c = new float[resultSize];
		if(resultSize < 2048) {
			down_sample(corr, c, resultSize, 2048/resultSize);
		} else {
			up_sample(corr, c, 2048, resultSize/2048);
		}
		ippsAdd_32f(results, c, results, resultSize);
		delete [] c;
	}
}